/*
* %Z%%W% %I%
*
===========================================================================
* Licensed Materials - Property of IBM
* "Restricted Materials of IBM"
* (C) Copyright IBM Corp. 2005. All Rights Reserved
*
===========================================================================
*/
package com.dovetail.jzos;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

/**
 * This class provides a static interface to various z/OS | OS/390
 * native library calls.
 */
public class ZUtil {

	public static final int LOG_ERR = 0;
	public static final int LOG_WARN = 1;
	public static final int LOG_NOTICE = 2;
	public static final int LOG_INFO = 3;
	public static final int LOG_DEBUG = 4;
	public static final int LOG_TRACE = 5;

	private static final String loggingLevels = "EWNIDT";

	//Constants that map the the native resource levels
	public static final int RESOURCE_READ = 1;
	public static final int RESOURCE_UPDATE = 2;
	public static final int RESOURCE_CONTROL = 3;
	public static final int RESOURCE_ALTER = 4;

	static {
		if(System.getProperty("java.version").startsWith("1.3")) {
			System.loadLibrary("jzos13");
		} else if (System.getProperty("java.version").startsWith("1.4")) {
			System.loadLibrary("jzos14");
		} else {
			System.loadLibrary("jzos50");
		}
		//Check to see if a logging level has been requested
		String logLevel = System.getProperty("jzos.logging");
		if (logLevel != null) {
		    int level = loggingLevels.indexOf(logLevel.toUpperCase().charAt(0));
		    if (level == -1) {
		        level = LOG_INFO;
		    }
		    setLoggingLevel(level);
		}		
	}

	// initialized to a constant default, but is set by launcher to match locale-specific (EBCDIC) default
	private static String defaultPlatformEncoding = ZFile.DEFAULT_EBCDIC_CODE_PAGE;

	private static class MvsCommandThread extends Thread {
		private long hCancellationECB;
		private MvsCommandCallback callback;
		//Placeholder for start parameters until a callback can be established.
		private String startParameters;
		
		/*
		 * Since a callback probably can't be established before a START command
		 * arrives, the parameters are cached and sent to the callback when set. 
		 */
		public void setCallback(MvsCommandCallback callback) {
		    this.callback = callback;
		    if (startParameters != null) {
		        callback.handleStart(startParameters);
		    }
		}
		public void run() {
			while (true) {
				//This call will block either until an MVS command is
				//received, or our shutdownHook is activated.
				String commandString = waitForMvsCommand(this);
				if (commandString == null) {
					//Shutdown hook processing, so we just return;
					return;
				} else {
				    char cmd = commandString.charAt(0);
				    commandString = commandString.substring(1);
				    switch (cmd) {
                    case 'P':
						wto("JZOS - MVS STOP command received", 0x0020, 0x4000);
						//Zero out the cancellationECB so our shutdown hook won't
						//try to POST it.
						hCancellationECB = 0;
						//Then bail nicely
						System.exit(0);                        
                        break;
                    case 'S':
						wto("JZOS - MVS START command received with parameters: " + commandString, 0x0020, 0x4000);
						startParameters = commandString;
                        if (callback != null) {
                            callback.handleStart(commandString);
                        }
                        break;
                    case 'F':
						wto("JZOS - MVS MODIFY command received: " + commandString, 0x0020, 0x4000);
                        if (callback != null) {
                            callback.handleModify(commandString);
                        }
                        break;
                    default:
						wto("JZOS - MVS <Unknown> command received", 0x0020, 0x4000);
                        break;
                    }
				}
			}
		}
	};
	
	private static MvsCommandThread mvsCommandThread;

	private ZUtil() {
	}

	/**
	 * Called by JZOSVM to allow this Java application intercept MVS Operator
	 * commands.
	 */
	public static void establishMvsCommandWaiter() {
	    mvsCommandThread = new MvsCommandThread();
	    mvsCommandThread.setDaemon(true);
		mvsCommandThread.start();
		Runtime.getRuntime().addShutdownHook(
				new Thread() {
					public void run() {
						if (mvsCommandThread.isAlive() && mvsCommandThread.hCancellationECB != 0) {
							cancelWaitForMvsCommand(mvsCommandThread.hCancellationECB);
						}
					}
				});
	}
	
	/**
	 * Register an MVS operator command callback.  This callback enables custom
	 * behavior based on the parameters received for START and MODIFY commands.
	 * STOP commmand processing is not customizable and always results in a call
	 * to System.exit().
	 * @param callback	an implementation of MvsCommandCallback that has customized
	 * 					handlers for START and MODIFY.
	 */
	public static void setMvsCommandCallback(MvsCommandCallback callback) {
	    if (mvsCommandThread != null && mvsCommandThread.isAlive()) {
	        mvsCommandThread.setCallback(callback);
	    } else {
	        throw new IllegalStateException("No mvsCallbackThread active - callback cannot be set");
	    }
	}
	
	/**
	 * Gets the default output encoding for this instance of the JVM.  The encoding
	 * is generally some variant of EBCDIC.   
	 * 
	 * @deprecated use getDefaultPlatformEncoding()
	 */
	public static String getDefaultOutputEncoding() {
		return getDefaultPlatformEncoding();
	}
	
	/**
	 * Set the default output encoding for this instance of the JVM.  The encoding
	 * is generally some variant of EBCDIC.
	 * 
	 * @deprecated use setDefaultPlatformEncoding()
	 * 
	 * @param encoding 	the charset encoding to use as a default
	 */
	public static void setDefaultOutputEncoding(String encoding) {
		setDefaultPlatformEncoding(encoding);
	}	
	
	/**
	 * Get the default platform encoding for this instance of the JVM.  The encoding
	 * is generally some variant of EBCDIC.  This property is set by the JZOS launcher
	 * to match the default locale encoding for the process.
	 * 
	 * @return 	the default output encoding
	 */
	public static String getDefaultPlatformEncoding() {
		return defaultPlatformEncoding;
	}
	
	/**
	 * Set the default output encoding for this instance of the JVM.  The encoding
	 * is generally some variant of EBCDIC. This property is set by the JZOS launcher
	 * to match the default locale encoding for the process.
	 * 
	 * @param encoding 	the charset encoding to use as a default
	 */
	public static void setDefaultPlatformEncoding(String encoding) {
		defaultPlatformEncoding = encoding;
	}	

	/**
	 * Get the value of the Environment variable.
	 * @param varName	the name of the environment variable
	 * @return 			the value of the environment variable.  Null if not found.
	 */
	public static native String getEnv(String varName);
	
	/**
	 * Sets the logging level for the native code.  This method can be used to
	 * capture debugging output from the ZFile and ZUtil native method
	 * @param level		one of the LOG_XXX constants defined above
	 */
	public static native void setLoggingLevel(int level);

	private static native String getJobname();

	/**
	 * Get the current MVS jobname, blanks trimmed
	 * @return the jobname
	 */
	public static String getCurrentJobname() {
	    return getJobname().trim();
	}

	private static native String getUsername() throws RcException;

	/**
	 * Get the current MVS user, blanks trimmed.
	 * @return the current MVS user
	 * 
	 * @throws RcException if the native call fails.
	 */
	public static String getCurrentUser() throws RcException {
	    return getUsername().trim();
	}

	/**
	 * Get a multi-line string that contains information about the version
	 * of Java that is currently running.  This method is called by the JZOSVM
	 * launcher to display the Java version information on the //SYSOUT DD
	 * 
	 * @return a String in the general form specified by sun.misc.Version.print()
	 */
	public static String getJavaVersionInfo() {
	    StringBuffer sb = new StringBuffer();
        /* First line: platform version. */
	    sb.append("java version \"");
	    sb.append(System.getProperty("java.version"));
	    sb.append("\"\n");
        /* Second line: runtime version (ie, libraries). */
	    sb.append(System.getProperty("java.runtime.name"));
	    sb.append(" (build ");
	    sb.append(System.getProperty("java.runtime.version"));
	    sb.append(")\n");
        /* Third line: JVM information. */
	    sb.append(System.getProperty("java.vm.name"));
	    sb.append(" (build ");
	    sb.append(System.getProperty("java.vm.version"));
	    sb.append(", ");
	    sb.append(System.getProperty("java.vm.info"));
	    sb.append(")");
	    return sb.toString();
	}

	/**
	 * Authenticate the user with the given password.  If authentication
	 * fails, an Exception is thrown.
	 * <p>
	 * This method calls the __passwd() C-Library function.
	 * @param user		the username to authenticate
	 * @param password	the user's credentials
	 * 
	 * @throws ErrnoException if authentication fails.  The errno and errno2 code
	 *         can be used to determine what the nature of the failure is.
	 */
	public static native void authenticate(String user, String password) throws ErrnoException;

	/**
	 * Determine if user has permission to the SAF entity.  If the authorization 
	 * check fails, an Exception is thrown.
	 * <p>
	 * This method calls the __check_resource_auth_np() C-Library function.
	 * @param user 			the username to authenticate
	 * @param safClass 		the SAF class (EJBROLE, FACILITY, etc...)
	 * @param safEntity		the (case sensitive) entity within the safClass (BPX.SERVER)
	 * @param resourceLevel	taken from the RESOURCE_XXX constants defined above
	 * 
	 * @throws ErrnoException if authorization fails.  The errno and errno2 code
	 *         can be used to determine what the nature of the failure is.
	 */
	public static native void hasPermission(String user, String safClass, String safEntity, int resourceLevel) throws ErrnoException;

	/**
	 * Issue a WTO.  This method automatically selects single line or multiline 
	 * based on the line length.
	 * @param message		a String, length > 125 characters is ignored
	 * @param routeCode		a bit mask of the 16 bits routing codes bits.
	 * @param descriptor	a bit mask of the 16 descriptor code bits.
	 * @return 				the WTO return code, 0 if successful.
	 */
	public static int wto(String message, int routeCode, int descriptor) {
	    if (message.length() <= 125) {
	        return WTO(message, routeCode, descriptor, getDefaultPlatformEncoding());
	    } else {
	        return MLWTO(formatLines(message, 70, "  "),
	                routeCode, descriptor, getDefaultPlatformEncoding());
	    }
	}

	/**
	 * Issue a multi-line WTO. All lines are supplied to the WTO macro using a
	 * linetype of "D".
	 * @param lines 		a array of up to 10 strings, length > 70 characters is
	 *        				ignored
	 * @param routeCode 	a bit mask of the 16 bits routing codes bits.
	 * @param descriptor 	a bit mask of the 16 descriptor code bits.
	 *
	 * @return 			the WTO return code, 0 if successful.
	 */
	public static int wto(String[] lines, int routeCode, int descriptor) {
	    return MLWTO(lines, routeCode, descriptor, getDefaultPlatformEncoding());
	}

	/**
	 * Format longString along word boundaries into an array of Strings that are each
	 * fewer than maxLength characters long.  Each word in a line will be delimited by
	 * a single space.  Lines 1..n (all but the first) will be indented by indent
	 * characters.  If an individual word is longer than maxLength, a message will be
	 * emitted to stderr and the word truncated to fit.
	 */
	static String[] formatLines(String longString, int maxLength, String indent) {
	    List lines = new ArrayList();
	    StringBuffer sb = new StringBuffer();
	    for (StringTokenizer strTok = new StringTokenizer(longString); strTok.hasMoreTokens(); ) {
	        String tok = strTok.nextToken();
	        if (sb.length() + tok.length() > maxLength) {
	            lines.add(sb.toString());
	            sb = new StringBuffer(indent);
	        }
	        if (tok.length() > maxLength) {
	            System.err.println("'" + tok + "' is too long for WTO line.  It will be truncated.");
	            sb.append(tok.substring(0,maxLength-sb.length()));
	        } else {
	            sb.append(tok);
	        }
	        //If we have room, append a space after the token.
	        if (sb.length() < maxLength) {
	            sb.append(" ");
	        }
	    }
	    if (sb.length() > 0) {
	        lines.add(sb.toString());
	    }
	    return (String[])lines.toArray(new String[lines.size()]);
	}

	private static native int MLWTO(String[] lines, 
	        							int routCde,
	                                    int desc,
	                                    String encoding);

	private static native int WTO(String message, 
	        						int routeCde, 
	        						int desc,
	        						String encoding);

	/**
	 * A convenience method to redirect the standard streams using the
	 * default character encoding.
	 * <p>
	 * JZOSVM calls this method on startup to cause System.out System.err and
	 * System.in to be directed to/from the job stream.
	 * 
	 * @throws Exception if the streams could not be redirected.
	 */
	public static void redirectStandardStreams() throws Exception {
	    redirectStandardStreams(getDefaultPlatformEncoding(), true);
	}
	
	/**
	 * A static method which redirects the standard streams as follows:
	 * <ul>
	 * <li>System.out is redirected to DD:STDOUT (required)
	 * <li>System.err is redirected to DD:STDERR (required)
	 * <li>System.in  is redirected from DD:STDIN (optional)
	 * </li>
	 * This method is invoked by the JZOSVM launcher to redirect the standard Java 
	 * streams to MVS DDs.  If STDIN is not defined, System.in is set to an
	 * InputStream that immediately returns -1 (EOS).
	 * @param requestedEncoding	the requested character encoding.  If this encoding
	 *        					is not supported, defaultPrintStreamEncoding is used 
	 * 							instead (Cp1047).  
	 * @return 					<code>true</code> if the requestedEncoding was used, 
	 * 							<code>false</code> if the default was used.
	 *   
	 * @throws Exception if the streams could not be redirected.
	 */
	public static boolean redirectStandardStreams(String requestedEncoding, final boolean enableTranscoding) throws Exception {

	    String effectiveEncoding = requestedEncoding;
		try {
			new String(new byte[1], requestedEncoding);
			//If the requestedEncoding is valid, make it the default
			setDefaultPlatformEncoding(requestedEncoding);
		} catch (UnsupportedEncodingException uee) {
		    effectiveEncoding = getDefaultPlatformEncoding();
		}

		final String enc = effectiveEncoding;
		try {
			AccessController.doPrivileged(new PrivilegedExceptionAction() {
			    public Object run() throws ZFileException {
					ZFile systemErrFile = new ZFile("DD:STDERR", "w");
					System.setErr(newEncodedPrintStream(systemErrFile.getOutputStream(), false, enc, enableTranscoding));
				
					ZFile systemOutFile = new ZFile("DD:STDOUT", "w");
					System.setOut(newEncodedPrintStream(systemOutFile.getOutputStream(), false, enc, enableTranscoding));
						
					//TODO should we also have an InputStream which translates bytes from desired encoding to default encoding?
					try {
						ZFile systemInFile = new ZFile("DD:STDIN", "r");
						System.setIn(systemInFile.getInputStream());
					} catch (Exception e) {
					    System.setIn(new InputStream() {
                            	public int read() throws IOException {
                            	    return -1;
                            	}
					        });
					}
					return null;
			    }
			});
		} catch (java.security.PrivilegedActionException pae) {
		    throw (Exception) pae.getException();
		}

		return effectiveEncoding.equals(requestedEncoding);
	}
	
	/**
	 * Construct a PrintStream constructed with the defaultOutputEncoding.
	 * @param os		the stream to base the PrintStream on.
	 * @param autoFlush	whether or not to autoFlush.
	 * @return 			the printStream
	 */
	public static PrintStream newEncodedPrintStream(OutputStream os, boolean autoFlush) {
		return newEncodedPrintStream(os, autoFlush, getDefaultPlatformEncoding());
	}

	/**
	 * Return a PrintStream constructed from the supplied OutputStream and the encoding.  
	 * We actually answer an instance of a subclass that also translates bytes sent 
	 * throught its OutputStream interface.
	 *  
	 * @param os 		the stream to base the PrintStream on.
	 * @param autoFlush	whether or not to autoFlush
	 * @param encoding	the charset encoding to use
	 * @return 			the printStream
	 * 
	 * @see TranscodingPrintStream
	 */
	public static PrintStream newEncodedPrintStream(OutputStream os, boolean autoFlush, String encoding) {
		return newEncodedPrintStream(os, autoFlush, encoding, true);
	}  

	/**
	 * Return a PrintStream constructed from the supplied OutputStream and the encoding.  
	 * We actually answer an instance of a subclass that also translates bytes sent 
	 * throught its OutputStream interface.
	 *  
	 * @param os 			the stream to base the PrintStream on.
	 * @param autoFlush		whether or not to autoFlush
	 * @param encoding		the charset encoding to use
	 * @param enable		whether or not to enable the transcoding
	 * @return 				the printStream
	 * 
	 * @see TranscodingPrintStream
	 */
	public static PrintStream newEncodedPrintStream(OutputStream os, boolean autoFlush, String encoding, boolean enable) {
	    TranscodingPrintStream tps = new TranscodingPrintStream(os, autoFlush, encoding, enable);
	    return tps;
	}  

	/*
	 * Natively set an instance's field with fieldObject.
	 */
	static native void setObjectField(Object instance, Object fieldValue, String fieldName, String fieldSig);

	/*
	 * Issue a JNI call that blocks until an MVS command arrives or
	 * is cancelled.
	 */
	private static native String waitForMvsCommand(Object mvsCommandThread);
	
	/*
	 * Issue a JNI call that causes an existing waitForMvsCommand to unblock
	 * This is issued by the shutdownHookThread if the waitForMvsCommandThead
	 * is active.
	 */
	private static native void cancelWaitForMvsCommand(long hCancellationECB);
}
