/*
* %Z%%W% %I%
*
===========================================================================
* Licensed Materials - Property of IBM
* "Restricted Materials of IBM"
* (C) Copyright IBM Corp. 2005. All Rights Reserved
*
===========================================================================
*/
package com.dovetail.jzos;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

/**
 * JNI Wrapper for z/OS | OS/390 C-Library IO routines.
 * An instance of ZFile is a thin wrapper around a C-library file handle and
 * can be used to access Datasets, DDs and HFS files. 
 * <p>
 * Java native methods in this class simply call their C-library equivalents. 
 * Please refer to the following IBM C/C++ publications, which describe how
 * to open MVS datasets and DD names in both record and stream mode.
 * <ul>
 * <li><a href="http://publibz.boulder.ibm.com/epubs/pdf/edclb131.pdf">
 * z/OS C/C++ Library Reference</a>
 * <li><a href="http://publibz.boulder.ibm.com/epubs/pdf/cbcpg130.pdf">
 * z/OS C/C++ Programming Guide</a>
 * </ul>
 * <i>Note:</i> By using this class to access HFS files, the file permissions built into the 
 * java.io.File class will be circumvented.  Therefore, this class requires that the
 * JzosPermission 'zfile.all' be granted by the policy in effect.  For now, this is
 * a simple binary check - either the ZFile class can be used or it can't.  Depending
 * on demand, this check could be refined for different types of files.
 * @see #ZFile(String, String) ZFile(String, String) for usage examples.
 */
public class ZFile {

	static {
		if(System.getProperty("java.version").startsWith("1.3")) {
			System.loadLibrary("jzos13");
		} else if (System.getProperty("java.version").startsWith("1.4")) {
			System.loadLibrary("jzos14");
		} else {
			System.loadLibrary("jzos50");
		}
	}
	
	/**
	 * Note: ZUtil.getDefaultPlatformEncoding() should be used to get the locale-specific
	 * EBCDIC default encoding for the current process.
	 */
	public static String DEFAULT_EBCDIC_CODE_PAGE = "Cp1047";
	public static String DEFAULT_ASCII_CODE_PAGE = "ISO8859-1";
	
    public static int LOCATE_RBA_EQ = 0;
    public static int LOCATE_KEY_FIRST = 1;
    public static int LOCATE_KEY_LAST = 2;
    public static int LOCATE_KEY_EQ = 3;
    public static int LOCATE_KEY_EQ_BWD = 4;
    public static int LOCATE_KEY_GE = 5;
    public static int LOCATE_RBA_EQ_BWD = 6;
    
    public static int SEEK_SET = 0;
    public static int SEEK_CUR = 1;
    public static int SEEK_END = 2;

    /** The name of the z/OS file to open. */
    private String name;
    /** The options, including z/OS extensions used to open the file. */
    private String options;
	/** Cached handle to information about the native file */
	private long hFileInfo;

	public static native void bpxwdyn(String command) throws RcException;
	
	/**
	 * Deletes an MVS dataset.
	 * <p>
	 * Constructs a fully qualified dataset name and calls the C-Library 
	 * remove() function.
	 * 
	 * @param datasetName  the dataset name to delete. If does not start with a single-quote, 
	 * then the current userid will be used as a high-level qualifier.
	 * @throws ZFileException
	 */
	public static void deleteDatatset(String datasetName) throws ZFileException {
		if (datasetName == null) 
			throw new NullPointerException("dataset name to delete may not be null");

		SecurityManager sm = System.getSecurityManager();
		if (sm != null) {
		    sm.checkPermission(new JzosPermission("zfile.all"));
		}	
		fremove("//'" + getFullyQualifiedDSN(datasetName) + "'");
	}
	
	/**
	 * Renames an MVS dataset.
	 * <p>
	 * Constructs a fully qualified dataset name and calls the C-Library 
	 * rename() function.
	 * 
	 * @param oldDatasetName  the dataset name to rename. If does not start with a single-quote, 
	 * then the current userid will be used as a high-level qualifier.

	 * @param newDatasetName  the new dataset name. If does not start with a single-quote, 
	 * then the current userid will be used as a high-level qualifier.
	 * 
	 * @throws ZFileException
	 */
	public static void renameDatatset(String oldDatasetName, String newDatasetName) throws ZFileException {
		if (oldDatasetName == null || newDatasetName == null) 
			throw new NullPointerException("dataset names to rename may not be null");

		SecurityManager sm = System.getSecurityManager();
		if (sm != null) {
		    sm.checkPermission(new JzosPermission("zfile.all"));
		}	
		frename("//'" + getFullyQualifiedDSN(oldDatasetName) + "'",
				"//'" + getFullyQualifiedDSN(newDatasetName) + "'");
	}
	
	public static native boolean ddExists(String ddname) throws ZFileException;

	public static boolean dsExists(String dsn) throws ZFileException {
		dsn = getFullyQualifiedDSN(dsn);
		String[] volumes = getVolumeList(dsn);
		if (volumes.length == 0) return false;
		String volume = volumes[0];
		if (volume.equals("MIGRAT")) return true;
		return readDSCB(dsn, volume, new byte[140]);	
	}
	
	public static native boolean hfsExists(String filename) throws ZFileException;
	
	public static boolean exists(String filename) throws ZFileException {
	    String upperFilename = filename.toUpperCase();
	    if (upperFilename.startsWith("//DD:") || upperFilename.startsWith("DD:")) {
	        String ddname = filename.substring(filename.indexOf(':')+1).trim();
	        return ddExists(ddname);
	    } else if (filename.startsWith("//") && !filename.startsWith("///")) {
	        String dsn = filename.substring(2);
	        return dsExists(dsn);
	    } else {
	        return hfsExists(filename);
	    }
	}

	/** 
	 * Returns a fully qualified dataset name if the given dsn is not quoted.
	 */
	public static String getFullyQualifiedDSN(String dsn) {  
		
		if (dsn.charAt(0) == '\'') { //is absolute name
			dsn = dsn.substring(1,dsn.length()-1);
		} else {
			try {
				dsn = ZUtil.getCurrentUser() + "." + dsn;
			} catch (RcException rc) {
				throw new RuntimeException ("Could not retrieve MVS_USERID " + rc.toString());
			}
		}
		return dsn;
	}
	
	/**
	 * Returns a String[] containing a list of volumes that contains
	 * the given fully-qualified dataset name.  
	 * Answers an empty array if the dataset is not found.
	 */
	public static String[] getVolumeList(String dsn) throws ZFileException {
		byte[] volBuf = new byte[265];
		if (locateDSN(dsn, volBuf)) {
			int volCount = ByteUtil.bytesAsInt(volBuf, 0, 2);
			String volumes[] = new String[volCount];
			int j=2;
			for (int i=0; i<volCount;i++) {
				try {
					String volser = new String(volBuf, j+4, 6, "Cp1047");
					volumes[i] = volser.trim();
				} catch (UnsupportedEncodingException uce) {
					throw new RuntimeException(uce.toString());
				}
				j = j+12;
			}
			return volumes;
		}
		return new String[0];  // not found
	}
	
	public static native boolean locateDSN(String dataSetName, byte[] volumeList) throws ZFileException;

	public static native boolean readDSCB(String dataSetName, String volume, byte[] volumeList) throws ZFileException;

    private static native void fremove(String fileName) throws ZFileException;
    
    private static native void frename(String oldName, String newName) throws ZFileException;

    /**
     * Constructor that creates a wrapper around a z/OS file based on the supplied name 
     * and options.  Cache the native file information for use on subsequent function calls.
     * <p>
     * This method calls the fopen() and fldata() C-library routines.
     * <p>
     * The following examples illustrate how to open various types of z/OS files:
     * <pre>ZFile f = new ZFile("/etc/profile", "r");</pre>
     * <p>Opens the HFS file /etc/profile for reading
     * <pre>ZFile dd = new ZFile("//DD:MYDD", "r");</pre>
     * <p>Opens the DD namee MYDD for reading
     * <pre>ZFile dsn = new ZFile("//'SYS1.HELP(ACCOUNT)'", "rt,type=record");</pre>
     * <p>Opens the member ACCOUNT from the PDS SYS1.HELP for reading text records
     * <pre>ZFile dsn = new ZFile("//SEQ", "wb,type=record,recfm=fb,lrecl=80");</pre>
     * <p>Opens the data set {MVS_USER}.SEQ for binary writing
     * <p>
     * The C-Library documentation should be consulted for complete details, which
     * can be quite involved.
     * <p>
     * @param name		the name of the native file to open
     * @param options	the options to use on the fopen() call
     * 
     * @throws ZFileException 		if the file could not be opened.
     * @throws SecurityException	if a SecurityManager is active and the required
     * 								JzosPermission is not granted based on the policy 
     * 								in effect.
     */
    public ZFile(String name, String options) throws ZFileException {
        this.name = name;
		this.options = options;
		
		SecurityManager sm = System.getSecurityManager();
		if (sm != null) {
		    sm.checkPermission(new JzosPermission("zfile.all"));
		}		
		//fopen(String, String) is private so that open is not
		//called on an already opened native file.  Use freopen() for
		//this purpose.
        hFileInfo = fopen(name, options);
    }

	/**
	 * Get the native file's blocksize.
     * <p>
     * This information is obtained via the fldata() C-Library routine
	 * @return the blocksize
	 * 
     * @throws ZFileException if the native call fails
	 */
    public int getBlksize() throws ZFileException {
        return getBlksize(getHFileInfo());
    }

    /**
     * Get the number of bytes read or written to the native file.
     * <p>
     * This information is obtained via the fldata() C-Library routine
     * @return the number of bytes read or written
     * 
     * @throws ZFileException if the native call fails
     */
    public long getByteCount() throws ZFileException {
    	return getByteCount(getHFileInfo());
    }

    /**
     * Get the native file's dataset organization.
     * <p>
     * This information is obtained via the fldata() C-Library routine
     * <p>
     * The possible organizations are:
     * <ul>
     * <li>Concat - Sequentially concateneated file
     * <li>Hiper - Memory file in Hiperspace
     * <li>HFS - Hierarchical File System file
     * <li>Mem - Memory file
     * <li>PDSE - Partitioned Data Set Extended file
     * <li>PDSmem - Partitioned data set member
     * <li>PDSdir - Partitioned data set directory
     * <li>PO - Partitioned data set
     * <li>VSAM - VSAM file
     * <li>PS - Sequential data set
     * <li>Temp - Temporary file
     * </ul>
     * @return the native file's DSORG
     * 
     * @throws ZFileException if the native call fails
     */
    public String getDsorg() throws ZFileException {
        return getDsorg(getHFileInfo());
    }

    /**
     * Gets the encoded position of the native file.
     * <p>
     * This method calls the fgetpos() C-library routine.
     * @return the native file position
     * 
     * @throws ZFileException if the native call fails
     */
    public byte[] getPos() throws ZFileException {
        return fgetpos(getHFileInfo());
    }
    
    /**
     * Get the name of the file.
     * @return the name of the native file
     */
    public String getFilename() {
        return name;
    }

	/**
	 * Get the file options.
	 * @return the options used to open the native file
	 */
	public String getOptions() {
		return options;
	}

    /**
     * Get an InputStream that can be used to read the native file.
     * <p>
	 * Prerequisite:  The native file was opened in streaming mode.
	 * @return a stream to be used for reading the native file
     */
    public InputStream getInputStream() {
		return new InputStream() {

			public void close() throws IOException {
				ZFile.this.close();
			}

			public int read() throws IOException {
				byte[] buf = new byte[1];
				int    nRead = ZFile.this.read(buf);

				if (nRead < 0) {
					return nRead;
				} else {
					return buf[0];
				}
			}

			public int read(byte[] buf, int off, int len) throws IOException {
				return ZFile.this.read(buf, off, len);
			}

			public int read(byte[] buf) throws IOException {
				return ZFile.this.read(buf);
			}
		};
    }

    /**
     * Get the native file's logical record length
     * @return the native file's LRECL
     * 
     * @throws ZFileException if the native call fails
     */
    public int getLrecl() throws ZFileException {
        return getLrecl(getHFileInfo());
    }

	/**
	 * Return an OutputStream that can be used to write the native file. 
	 * <p> 
	 * Prerequisite:  The native file was opened in streaming mode.
	 * @return a stream to be used for writing to the native file
	 */
    public OutputStream getOutputStream() {
		return new OutputStream() {

				public void close() throws IOException {
					ZFile.this.close();
				}

				public void flush() throws IOException {
					ZFile.this.flush();
				}

				public void write(int b) throws IOException {
    	
					byte[] buf = new byte[1];
					buf[0] = (byte)b;
					ZFile.this.write(buf);
				}

				public void write(byte[] b, int off, int len) throws IOException {
				   ZFile.this.write(b, off, len);
				}

				public void write(byte[] b) throws IOException {
					ZFile.this.write(b);
				}
			};
    }

    /**
     * Get the native file's record format.
     * <p>
     * This information is obtained via the fldata() C-Library routine
     * <p>
     * The first character of the returned string will be one of the following:
     * <ul>
     * <li>F - Fixed length records
     * <li>V - Variable length records
     * <li>U - Undefined length records
     * </ul>
     * <p>
     * Subsequent characters may also be present in the returned String:
     * <ul>
     * <li>A - The file has ASA print control characters
     * <li>B - The file has Blocked records
     * <li>M - The file has Machine print control characters
     * <li>S - The file has Standard (if Fixed) or Spanned (if Variable) records
     * </ul>
     * @return the native file's RECFM
     * 
     * @throws ZFileException if the native call fails
     */
    public String getRecfm() throws ZFileException {
        return getRecfm(getHFileInfo());
    }

    /**
     * Get the number of records read/written.  
     * <p>
     * Prerequisite: The native file was opened in record mode.
     * @return the number of records read or written
     * 
     * @throws ZFileException if the native call fails
     */
    public long getRecordCount() throws ZFileException {
    	return getRecordCount(getHFileInfo());
    }

    /**
     * Return the VSAM type.
     * <p>
     * This information is obtained via the fldata() C-Library routine
     * <p>
     * The possible VSAM types are:
     * <ul>
     * <li>ESDS
     * <li>KSDS
     * <li>RRDS
     * <li>ESDS_PATH
     * <li>KSDS_PATH
     * <li>NOTVSAM
     * </ul>
     * @return the native file's VSAM type
     * 
     * @throws ZFileException if the native call fails
     */
    public String getVsamType() throws ZFileException {
        return getVsamType(getHFileInfo());
    }

    /**
     * Close the file.  Regardless of the success of the native operation,
     * set the handle to 0 (null) so that the file is rendered invalid.
     * <p>
     * This method calls the fclose() C-Library routine.
     * 
     * @throws ZFileException if the native call fails
     */
    public void close() throws ZFileException {
    	try {
        	fclose(getHFileInfo());
    	} finally {
    		hFileInfo = 0;
    	}
    }

    /**
     * Flush the native file.
     * <p>
     * This method calls the fflush() C-library routine.
     * 
     * @throws ZFileException if the native call fails
     */
    public void flush() throws ZFileException {
        fflush(getHFileInfo());
    }

    /**
     * Locate a record using its record number. 
     * <p>
     * Prerequisite:  The native file is VSAM.
     * <p>
     * This method calls the flocate() C-library routine.
     * @param recordNumber 	the VSAM record number to locate
     * @param options		the locate options to use
     * 
     * @throws ZFileException if the native call fails
     */
    public void locate(int recordNumber, int options) throws ZFileException {
        locate(ByteUtil.intAsBytes(recordNumber), options);
    }

    /**
     * Locate a record given a key. 
     * <p>
     * Prerequisite:  The native file is VSAM.
     * <p>
     * This method calls the flocate() C-library routine
     * @param key		the VSAM key to use for the flocate() call
     * @param options	the locate options to use
     * 
     * @throws ZFileException if the native call fails
     */
    public void locate(byte[] key, int options) throws ZFileException {
        flocate(getHFileInfo(), key, options);
    }

    /**
     * Read a record from the file into a buffer.
     * <p>
     * This method calls the fread() C-library routine, 
     * but return code is made to be compatible with Stream.read().
     * @param buf	the byte array into which the bytes will be read
     * @return 		the number of bytes read, -1 if EOF encountered.
     * 
     * @throws ZFileException if the native call fails
     */
    public int read(byte[] buf) throws ZFileException {
        return fread(getHFileInfo(), buf, 0, buf.length);
    }

    /**
     * Read from the native file into the supplied buffer.
     * <p>
     * This method calls the fread() C-library routine, 
     * but return code is made to be compatible with Stream.read().
     * @param buf		the byte array into which the bytes will be read
     * @param offset	the offset, inclusive in buf to start reading bytes
     * @param len		the number of bytes to read
     * @return 			the number of bytes read, -1 if EOF encountered.
     * 
     * @throws ZFileException if the native call fails
     */
    public int read(byte[] buf, int offset, int len) throws ZFileException {
        return fread(getHFileInfo(), buf, offset, len);
    }

    /**
     * Reopen the file with different mode options.
     * <p>
     * This method calls the freopen() C-library routine.
     * @param options	the new options to reopen the file with
     * 
     * @throws ZFileException if the native call fails
     */
    public void reopen(String options) throws ZFileException {
        freopen(getHFileInfo(), options);
    }

    /** 
     * Seeks the file to the beginning and resets the counts.
     * <p>
     * This method calls the rewind() C-library routine.  After successful
     * execution getByteCount() and getRecordCount() will return 0.
     * 
     * @throws ZFileException if the native call fails
     */
    public void rewind() throws ZFileException {
        rewind(getHFileInfo());
    }
    /**
     * Seek the file to the specified offset from origin.
     * <p>
     * This method calls the fseeko() C-library routine.
     * @param offset	from origin.  Refer to the
     *                  C documentation for details regarding
     *                  record based files.
     * @param origin    where to offset from.  SEEK_SET, SEEK_CUR, SEEK_END
     * 
     * @throws ZFileException if the native call fails
     */
    public void seek(long offset, int origin) throws ZFileException {
        fseeko(getHFileInfo(), offset, origin);
    }
    
    /**
     * Sets the position of the native file.
     * <p>
     * This method calls the fsetpos() C-library routine.
     * @param position	the native file position, which was obtained from
     *                 	a prior getPos() call.
     * 
     * @throws ZFileException if the native call fails
     */
    public void setPos(byte[] position) throws ZFileException {
        fsetpos(getHFileInfo(), position);
    }
    /**
     * Returns the position of the file.
     * <p>
     * This method calls the ftello() C-library routine.
     * @return	The file position.  Refer to the C documentation for
     * 			details regarding record based files.
     * 
     * @throws ZFileException if the native call fails
     */
    public long tell() throws ZFileException {
        return ftello(getHFileInfo());
    }
    
    /**
     * Update the current record of the native file.
     * <p>
     * Prerequisite: The native file was opened in record mode.
     * <p>
     * This method calls the fupdate() C-library routine.
     * @param buf 	the byte array to use to update the current record with
     * @param size	the size of the record
     * @return 		the number of bytes written.
     * 
     * @throws ZFileException if the native call fails
     */
    public int update(byte[] buf, int size) throws ZFileException {
        return fupdate(getHFileInfo(), buf, size);
    }
    
    /**
     * Delete the current VSAM record.
     * <p>
     * Prerequisite: The native VSAM file was opened in record mode
     * <p>
     * This method calls the fdelrec() C-library routine.
     * 
     * @throws ZFileException if the native call fails
     */
    public void delrec() throws ZFileException {
        fdelrec(getHFileInfo());
    }    

    /**
     * Write a record to the native file.
     * <p>
     * Prerequisite: The native file was opened in record mode.
     * <p>
     * This method calls the fwrite() C-library routine.
     * @param buf	the byte array to write to the native file
     * 
     * @throws ZFileException if the native call fails
     */
    public void write(byte[] buf) throws ZFileException {
        fwrite(getHFileInfo(), buf, 0, buf.length);
    }

    /**
     * Write the buffer to the native file.
     * <p>
     * This method calls the fwrite() C-library routine.
     * @param buf		the byte array to write
     * @param offset	the offset, inclusive in buf to start writing bytes from
     * @param len		the number of bytes to write
     * 
     * @throws ZFileException if the native call fails
     */
    public void write(byte[] buf, int offset, int len) throws ZFileException {
        fwrite(getHFileInfo(), buf, offset, len);
    }
    
	/**
	 * Return the cached handle to the native file information.
	 * @throws IllegalStateException if this has not been set
	 */
	private long getHFileInfo() throws IllegalStateException {
		if (hFileInfo == 0) {
			throw new IllegalStateException("There is no native file information");
		}
		return hFileInfo;
	}

    private native void fclose(long hFile) throws ZFileException;
    private native void fflush(long hFile) throws ZFileException;
    private native byte[] fgetpos(long hfile) throws ZFileException;
    private native void flocate(long hFile, byte[] key, int options) throws ZFileException;
    private native long fopen(String name, String options) throws ZFileException;
    private native int fread(long hFile, byte[] buf, int offset, int len) throws ZFileException;
    private native void freopen(long hFile, String options) throws ZFileException;
    private native void rewind(long hFile) throws ZFileException;
    private native void fseeko(long hFile, long offset, int origin) throws ZFileException;
    private native void fsetpos(long hFile, byte[] position) throws ZFileException;
    private native long ftello(long hFile) throws ZFileException;
    private native int fupdate(long hFile, byte[] buf, int size) throws ZFileException;
    private native void fdelrec(long hFile) throws ZFileException;
    private native void fwrite(long hFile, byte[] buf, int size, int count) throws ZFileException;
    private native int getBlksize(long hFile) throws ZFileException;
    private native long getByteCount(long hFile) throws ZFileException;
    private native String getDsorg(long hFile) throws ZFileException;
    private native int getLrecl(long hFile) throws ZFileException;
    private native String getRecfm(long hFile) throws ZFileException;
    private native long getRecordCount(long hFile) throws ZFileException;
    private native String getVsamType(long hFile) throws ZFileException;
}
