/*
* %Z%%W% %I%
*
===========================================================================
* Licensed Materials - Property of IBM
* "Restricted Materials of IBM"
* (C) Copyright IBM Corp. 2005. All Rights Reserved
*
===========================================================================
*/
package com.dovetail.jzos;

import java.io.BufferedWriter;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;


/**
 * A subclass of PrintStream that sets an encoding on its OutputStreamWriter
 * and also transcodes any write(byte[]) messages that are sent to it
 * when it is treated as an OutputStream.  This avoids the problem caused 
 * an application wraps System.out or System.err in its own PrintWriter which
 * would override and lose the PrintStream's encoding.
 */
public class TranscodingPrintStream extends PrintStream {

    private boolean active = true;
    private byte[] translateTable;
    private byte[] translateBuffer = new byte[8092];
    
    /**
     * Create a new print stream.
     *
     * @param  out        The output stream to which values and objects will be
     *                    printed
     * @param  autoFlush  A boolean; if true, the output buffer will be flushed
     *                    whenever a byte array is written, one of the
     *                    <code>println</code> methods is invoked, or a newline
     *                    character or byte (<code>'\n'</code>) is written
     *
     * @param encoding    The String naming the encoding to use, both
     *                    for the internal OutputStreamWriter and for
     *                    translating write(byte[]) IOStream messages
     *                    sent to this object
     *  
     * @see java.io.PrintStream#PrintStream(java.io.OutputStream, boolean)
     * @see java.io.PrintWriter#PrintWriter(java.io.OutputStream, boolean)
     */
    public TranscodingPrintStream(OutputStream out, boolean autoFlush, String encoding, boolean enable) {
        super(out, autoFlush);
        this.active = enable; 
        initializeEncoding(out, encoding);
    }
    
    /**
     * Initialize the encoding for our PrintStream.
	 */
    private void initializeEncoding(OutputStream out, String encoding) {

        OutputStreamWriter osw = null;
		try {
		    // here's the encoding OutputStreamWriter that we will be using for char/string printing.
			osw = new OutputStreamWriter(out, encoding);
			
			// build a byte-to-byte transcoding table used for translating
			// writes to this object's OutputStream protocol
			// But if the default and target encodings are equivalent, 
			// then set it to null to indicate no translation of bytes.
			if (active) {  //If transcoding was not requested, don't build table.
				byte[] indentityMap = new byte[256];
				for (int i=0; i<indentityMap.length; i++ ) indentityMap[i] = (byte)i;
				if ((new String(indentityMap)).equals(new String(indentityMap, encoding))) {
					translateTable = null;
				} else {
					translateTable = (new String(indentityMap)).getBytes(encoding);
					if (translateTable.length != indentityMap.length) {
						translateTable = null;  // Disable transcoding.
						ZUtil.wto("JZOS - Both the default file.encoding and the encoding " 
								+ encoding 
								+ " must be single byte encodings; output transcoding is disabled",
								0x0020, 0x4000);
					}
				}
			}
	 	} catch (UnsupportedEncodingException oee) {
	 		// wto can be removed if we can figure out why exceptions are described in JNI
	 		ZUtil.wto("JZOS - The encoding " + encoding + " is not supported: " + oee.toString(),
	 				0x0020, 0x4000);  
			throw new RuntimeException("The encoding " + encoding + " is not supported: " + oee.toString());
		}
	 	
	    // Since there is no interface in JDK1.3 PrintStream for setting encoding, 
	    // we are forced into unnatural acts...
		BufferedWriter bw = new BufferedWriter(osw);
		ZUtil.setObjectField(this, osw, "charOut", "Ljava/io/OutputStreamWriter;");
		ZUtil.setObjectField(this, bw, "textOut", "Ljava/io/BufferedWriter;");
    }
  
    /**
     * Override {@link java.io.PrintStream#write(int b)}
     * to transcode bytes from the default encoding to the target encoding.
     */
    public void write(int b) {
	
        if (isActive()) {
            super.write(translateTable[b & 0xFF] & 0xFF);
        } else {
            super.write(b);
        }    
    }

    /**
     * Override {@link java.io.PrintStream#write(byte[], int, int)}
     * to transcode bytes from the default encoding to the target encoding.
     */
    public void write(byte buf[], int off, int len) {
        
        if (isActive()) {
            writeTranslatedBytes(buf, off, len);
        } else {
            super.write(buf, off, len);
        }    
    }
    
    /**
     * Translate bytes from a byte array and write them to the receiver's
     * output stream.  Synchronized access to the reusable translateBuffer.
     */
    private synchronized void writeTranslatedBytes(byte buf[], int off, int len) {
        if (len > translateBuffer.length) {
            translateBuffer = new byte[len * 5 / 4];
        }
        for (int i=0; i<len; i++ ) {
            translateBuffer[i] = translateTable[buf[i+off] & 0xFF];
        }
        super.write(translateBuffer, 0, len);
    }
        
    /**
     * @return whether this PrintStream is actively transcoding.  If false, the write()
     * methods default to the superclass behavior.
     */
    public boolean isActive() {
        return active && translateTable != null;
    }
}
