/*
* %Z%%W% %I%
*
===========================================================================
* Licensed Materials - Property of IBM
* "Restricted Materials of IBM"
* (C) Copyright IBM Corp. 2005. All Rights Reserved
*
===========================================================================
*/
package com.dovetail.jzos;

import java.io.IOException;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.NoSuchElementException;


/**
 * @author Stephen Goetze
 */
public class PdsDirectory {
    /*--------------------------------------------------------------------+

    Ref: MVS/XA Data Administration Guide, IBM Publication GC26-4013

    A Partitioned Data Set Directory Block

    +----------+--------------------+--------------------+-------------...
    |Number of |    Member Entry    |    Member Entry    |    Member En
    |Bytes Used|         A          |         B          |           C
    |(Maximum  |                    |                    |
    |  256)    |                    |                    |
    +----------+--------------------+--------------------+-------------...
    <----2-----><----------------Maximum 254----------------------------->

    A Partitioned Data Set Member Entry

    +----------------+------+--+-----------------------------------------+
    |     Member     |  TTR |C |             Optional User Data          |
    |      Name      |      |  |  TTRN  |  TTRN    |  TTRN   |           |
    +----------------+------+--+-----------------------------------------+
    <-------8--------><--3--><1><------------0 to 31 halfwords----------->

    Data in Member Entry 'C' Field Bitfield Definitions
    +-----------+-------------+----------------------+
    |   1 if    |  Number of  |    Number of User    |
    |Name is an |  User Data  |    Data Halfwords    |
    |  Alias    |    TTRNs    |                      |
    +-----------+-------------+----------------------+
        0           1-2               3-7
   +--------------------------------------------------------------------*/
    private class PdsDirectoryIterator implements Iterator {
    
        private byte[] cb = new byte[256];
        int pos;
        private int blen;
        
        public PdsDirectoryIterator() {
            getNextBlock();
        }

        private void getNextBlock() {
            try {
                if (directory.read(cb) == 0) {
                    throw new IllegalStateException("Expecting another directory block, but found none");
                }
            } catch(IOException ioe) {
                throw new RuntimeException("IOException occurred attempting to read next directory block: " + ioe.getMessage() );
            }
            blen = ByteUtil.bytesAsInt(cb, 0, 2);
            pos = 2; //Point to beginning of first member
        }

        public boolean hasNext() {
            //First check to see if we are at the end of the current block.
            //If so, get the next block.
            if ((pos+1) >= blen) {
                getNextBlock();
            }
            //Now check the next member name.  If it is 8 consecutive '0xFF'
            //bytes, there are no more members.
            for (int i = pos; i < (pos+8); i++ ) {
                if (cb[i] != (byte)0xFF) {
                    return true;
                }
            }
            return false;
        }

        public Object next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            MemberInfo mi = new MemberInfo(cb, pos);
            //Advance pos to the next member area, or the end of the block.
            pos += mi.getSizeInBytes();
            return mi;
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }
        
    }
    
    public static class MemberInfo {
        /*
         * The layout for these SPF statistics can be found in the
         * ISPF Developer's Guide and Reference.
         * Byte Description and Format (modified to use 0 offset)
         * 0 Version number, in hexadecimal format. Value is between X'01' and X'99'.
         * 1 Modification level, in hexadecimal format. Value is between X'00' and X'99'.
         * 2 Flags:
         *  Bit 1 SLCM indicator. SCLM uses this to determine whether the member
         *  and any related SCLM information are still in sync.
         *  v ON means the member was last edited by SCLM, the PDF
         *  Software Configuration and Library Manager.
         *  v OFF means the member was somehow processed outside SCLM.
         *  Bit 2�8 Reserved for future ISPF use.
         * 3 The seconds portion of the time last modified, in packed decimal format.
         * 4�7 Creation date:
         *  Byte 4 Century indicator.
         *  Byte 5�7 Julian date, in packed decimal format
         * 8�11 Date last modified:
         *  Byte 8 Century indicator.
         *  Byte 9�11 Julian date, in packed decimal format
         * 12�13 Time last modified, in packed format:
         *  Byte 12 Hours, in packed decimal format
         *  Byte 13 Minutes, in packed decimal format
         * 14�15 Current number of lines, in hexadecimal format
         * 16�17 Initial number of lines, in hexadecimal format
         * 18�19 Number of modified lines, in hexadecimal format
         * 20�26 Userid, in character format
         * 27�29 Reserved for future ISPF use 
         */
        public class Statistics {
            public int version;
            public int mod;
            public byte flags;
            public Date creationDate;
            public Date modificationDate;
            public int currentLines;
            public int initialLines;
            public int modifiedLines;
            public String userid;
            public Statistics() throws RuntimeException {
                byte[] ud = MemberInfo.this.userdata;
                if (ud.length != 30) {
                    throw new IllegalStateException("Userdata not 30 bytes long");
                }
                //One of the following statements may throw a RuntimeException
                //if we are not really working with statistics info
                //This is handled by the caller
                version = ByteUtil.bytesAsInt(ud, 0, 1);
                mod = ByteUtil.bytesAsInt(ud, 1, 1);
                flags = ud[2];
                Calendar cal = Calendar.getInstance();
                cal.clear();
                cal.set(Calendar.YEAR, (int)ByteUtil.unpackLong(ud, 5, 1, false));
                cal.set(Calendar.DAY_OF_YEAR, (int)ByteUtil.unpackLong(ud, 6, 2, true));
                creationDate = cal.getTime();
                cal.clear();
                cal.set(Calendar.YEAR, (int)ByteUtil.unpackLong(ud, 9, 1, false));
                cal.set(Calendar.DAY_OF_YEAR, (int)ByteUtil.unpackLong(ud, 10, 2, true));
                cal.set(Calendar.HOUR_OF_DAY, (int)ByteUtil.unpackLong(ud, 12, 1, false));
                cal.set(Calendar.MINUTE, (int)ByteUtil.unpackLong(ud, 13, 1, false));
                cal.set(Calendar.SECOND, (int)ByteUtil.unpackLong(ud, 3, 1, false));
                modificationDate = cal.getTime();
                currentLines = ByteUtil.bytesAsInt(ud, 14, 2);
                initialLines = ByteUtil.bytesAsInt(ud, 16, 2);
                modifiedLines = ByteUtil.bytesAsInt(ud, 18, 2);
                userid = new String(ud, 20, 7);
            }
            public String toString() {
                StringBuffer sb = new StringBuffer();
                sb.append(version);
                sb.append(".");
                sb.append(mod);
                sb.append(" Created: ");
                sb.append(creationDate);
                sb.append(" Last Modified: ");
                sb.append(modificationDate);
                sb.append(" Current Lines: ");
                sb.append(currentLines);
                sb.append(" Intial Lines: ");
                sb.append(initialLines);
                sb.append(" Modified Lines: ");
                sb.append(modifiedLines);
                sb.append(" Userid: ");
                sb.append(userid);
                return sb.toString();
            }
        }
        private String name;
        private byte cByte;
        private byte[] userdata;
        private Statistics statistics;
        
        public MemberInfo(byte[] bytes, int offset) {
            name = new String(bytes, offset, 8);
            cByte = bytes[offset+11];
            //Determine the user data length.  Bits 3-7 of the 'C' byte
            //(offset 12 from the beginning of the member data) contain the number
            //of halfwords of userdata.  This number is multiplied by 2 to get the
            //number of bytes.
            int userDataLen = (cByte & 0x1F) * 2;
            //System.out.println("userDataLen= " + userDataLen);
            userdata = new byte[userDataLen];
            for (int i=0; i<userDataLen; i++) {
                //The userdata, if present, starts at byte 13
                userdata[i] = bytes[offset+12+i];
            }
            //Try to read the userdata as SPF statistics.  If they can't be
            //determined, leave the iv null.
            try {
                statistics = new Statistics();
            } catch(IllegalStateException ise) {
                //Eat it, and leave statistics null
            }
        }
        
        public boolean isAlias() {
            return (cByte & 0x80) != 0;
        }
        
        public int getSizeInBytes() {
            //System.out.println("getSizeInBytes= " + (12 + userdata.length));
            return 12 + userdata.length; 
        }
        
        public byte[] getUserData() {
            return userdata;
        }
        
        public Statistics getStatistics() {
            return statistics;
        }
        
        public String toString() {
            StringBuffer sb = new StringBuffer(name);
            if (statistics != null) {
                sb.append(" ");
                sb.append(statistics.toString());
            }
            return sb.toString();
        }
    }
    private ZFile directory;

    public PdsDirectory(String pdsName) throws IOException {
        directory = new ZFile(pdsName, "rb,recfm=u,type=record,blksize=256");
        if (!directory.getDsorg().equals("PDSdir")) {
            throw new IOException("The name " + pdsName + " does not represent a PDS Directory");
        }
    }
    
    public void close() throws IOException {
        directory.close();
    }
    
    public Iterator iterator() {
        return new PdsDirectoryIterator();
    }
}
