/*
* %Z%%W% %I%
*
===========================================================================
* Licensed Materials - Property of IBM
* "Restricted Materials of IBM"
* (C) Copyright IBM Corp. 2005. All Rights Reserved
*
===========================================================================
*/
package com.dovetail.jzos;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;

/**
 * A class with static methods for building a BufferedReader, BufferedWriter,
 * InputStream, or OutputStream on a text file. Clients of this class
 * <p>
 * If the file name given starts with "//", then the underlying file stream will
 * be constructed using a ZFile on an MVS dataset, otherwise the file will be
 * constructed using java.io.
 * <p>
 * The default encoding for BufferedReaders and BufferedWriters is the 
 * ZUtil.defaultPlatformEncoding for ZFiles, and the default JVM file.encoding 
 * property for other files.
 */
public class FileFactory {

	/**
	 * Should never create an instance of this class.
	 */
	private FileFactory() {
	}
	
	/**
	 * Open a new unbuffered InputStream on a file with the given name.
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this InputStream.
	 */
	public static InputStream newInputStream(String filename) throws IOException {
		if (filename == null)
			throw new IllegalArgumentException("filename may not be null");
		if (filename.startsWith("//")) {
			return makeZFileInputStream(filename);
		} else {
			return makeJavaIoInputStream(filename);
		}
	}
	
	/**
	 * Open a new BufferedInputStream on a file with the given name.
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this InputStream.
	 */
	public static BufferedInputStream newBufferedInputStream(String filename, int bsize) throws IOException {
		return new BufferedInputStream(newInputStream(filename), bsize);
	}

	/**
	 * Open a new BufferedInputStream on a file with the given name.
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this InputStream.
	 */
	public static BufferedInputStream newBufferedInputStream(String filename) throws IOException {
		return new BufferedInputStream(newInputStream(filename));
	}

	/**
	 * Open a new unbuffered OutputStream on a file with the given name.
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this OutputStream.
	 */
	public static OutputStream newOutputStream(String filename) throws IOException {
		if (filename == null)
			throw new IllegalArgumentException("filename may not be null");
		if (filename.startsWith("//")) {
			return makeZFileOutputStream(filename);
		} else {
			return makeJavaIoOutputStream(filename);
		}
	}

	/**
	 * Open a new BufferedOutputStream on a file with the given name and buffer size.
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this OutputStream.
	 */
	public static BufferedOutputStream newBufferedOutputStream(String filename, int bsize) throws IOException {
		return new BufferedOutputStream(newOutputStream(filename), bsize);
	}

	/**
	 * Open a new BufferedOutputStream on a file with the given name.
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this OutputStream.
	 */
	public static BufferedOutputStream newBufferedOutputStream(String filename) throws IOException {
		return new BufferedOutputStream(newOutputStream(filename));
	}
	

	/**
	 * Open a new BufferedReader on a file with the given name, encoding, and
	 * buffersize. 
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this reader.
	 */
	public static BufferedReader newBufferedReader(String filename, String encoding,
			int bsize) throws IOException {
		if (filename == null)
			throw new IllegalArgumentException("filename may not be null");
		if (filename.startsWith("//")) {
			return makeZFileReader(filename, encoding, bsize);
		} else {
			return makeJavaIoReader(filename, encoding, bsize);
		}
	}

	/**
	 * Open a new BufferedReader on a file with the given name, encoding, and
	 * a  default buffersize.
	 * <p> 
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this reader.
	 */
	public static BufferedReader newBufferedReader(String filename, String encoding) throws IOException {
		return newBufferedReader(filename, encoding, -1);
	}
	
	/**
	 * Open a new BufferedReader on a file with the given name, default encoding, and
	 * a  default buffersize. 
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this reader.
	 */
	public static BufferedReader newBufferedReader(String filename) throws IOException {
		return newBufferedReader(filename, null, -1);
	}	

	/**
	 * Open a new BufferedWriter on a file with the given name, encoding, and
	 * buffersize. 
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this reader.
	 */
	public static BufferedWriter newBufferedWriter(String filename, String encoding,
			int bsize) throws IOException {
		if (filename == null)
			throw new IllegalArgumentException("filename may not be null");
		if (filename.startsWith("//")) {
			return makeZFileWriter(filename, encoding, bsize);
		} else {
			return makeJavaIoWriter(filename, encoding, bsize);
		}
	}

	/**
	 * Open a new BufferedWriter on a file with the given name, encoding, and
	 * default buffersize. 
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this reader.
	 */
	public static BufferedWriter newBufferedWriter(String filename, String encoding ) throws IOException {
		if (filename == null)
			throw new IllegalArgumentException("filename may not be null");
		if (filename.startsWith("//")) {
			return makeZFileWriter(filename, encoding, -1);
		} else {
			return makeJavaIoWriter(filename, encoding, -1);
		}
	}

	/**
	 * Open a new BufferedWriter on a file with the given name, default encoding, and
	 * default buffersize. 
	 * <p>
	 * File names starting with "//" are assumed to be MVS datasets
	 * and are opened using ZFile. Others are opened with regular java.io files.
	 * <p>
	 * NB: the client is responsible for closing this reader.
	 */
	public static BufferedWriter newBufferedWriter(String filename ) throws IOException {
		if (filename == null)
			throw new IllegalArgumentException("filename may not be null");
		if (filename.startsWith("//")) {
			return makeZFileWriter(filename, null, -1);
		} else {
			return makeJavaIoWriter(filename, null, -1);
		}
	}

	/**
	 * Answer the default encoding assumed for a given filename
	 */
	public static String getDefaultEncoding(String filename) {
		if (filename.startsWith("//")) {
			return getDefaultZFileEncoding();
		} else {
			return getDefaultJavaIoEncoding();
		}
	}

	/**
	 * Answer the default (EBCDIC) encoding for a ZFile
	 */
	public static String getDefaultZFileEncoding() {
		return ZUtil.getDefaultPlatformEncoding();
	}

	/**
	 * Answer the default encoding for a java.io file encoding, 
	 * which defaults to the Java System property "file.encoding". 
	 */
	public static String getDefaultJavaIoEncoding() {
		return System.getProperty("file.encoding", "ISO-8859-1");
	}
	
	
	private static BufferedReader makeJavaIoReader(String filename,
			String encoding, int bsize) throws IOException {
		InputStream is = new FileInputStream(filename);
		if (encoding == null) {
			if (bsize <= 0) {
				return new BufferedReader(new InputStreamReader(is));
			} else {
				return new BufferedReader(new InputStreamReader(is), bsize);
			}			
		} else {
			if (bsize <= 0) {
				return new BufferedReader(new InputStreamReader(is, encoding));
			} else {
				return new BufferedReader(new InputStreamReader(is, encoding), bsize);
			}				
		}
	}

	private static BufferedWriter makeJavaIoWriter(String filename,
			String encoding, int bsize) throws IOException {

		OutputStream os = new FileOutputStream(filename);
		if (encoding == null) {
			if (bsize <= 0) {
				return new BufferedWriter(new OutputStreamWriter(os));
			} else {
				return new BufferedWriter(new OutputStreamWriter(os), bsize);
			}
		} else {
			if (bsize <= 0) {
				return new BufferedWriter(new OutputStreamWriter(os, encoding));
			} else {
				return new BufferedWriter(new OutputStreamWriter(os, encoding), bsize);
			}			
		}
	}
	
	private static BufferedReader makeZFileReader(String filename,
			String encoding, int bsize) throws IOException {
		ZFile zFile = new ZFile(filename, "rt");
		InputStream is = zFile.getInputStream();
		if (encoding == null)
			encoding = getDefaultZFileEncoding();
		if (bsize <= 0) {
			return new BufferedReader(new InputStreamReader(is, encoding));
		} else {
			return new BufferedReader(new InputStreamReader(is, encoding), bsize);
		}			
	}

	private static BufferedWriter makeZFileWriter(String filename,
			String encoding, int bsize) throws IOException {
		ZFile zFile = new ZFile(filename, "wt");
		OutputStream os = zFile.getOutputStream();
		if (encoding == null)
			encoding = getDefaultZFileEncoding();
		if (bsize <= 0) {
			return new BufferedWriter(new OutputStreamWriter(os, encoding));
		} else {
			return new BufferedWriter(new OutputStreamWriter(os, encoding), bsize);
		}		
	}

	private static InputStream makeZFileInputStream(String filename) throws IOException {
		ZFile zFile = new ZFile(filename, "rt");
		return zFile.getInputStream();
	}	

	private static OutputStream makeZFileOutputStream(String filename) throws IOException {
		ZFile zFile = new ZFile(filename, "wt");
		return zFile.getOutputStream();
	}	

	private static InputStream makeJavaIoInputStream(String filename) throws IOException {
		return new FileInputStream(filename);
	}	

	private static OutputStream makeJavaIoOutputStream(String filename) throws IOException {
		return new FileOutputStream(filename);
	}	

}