/*
* %Z%%W% %I%
*
===========================================================================
* Licensed Materials - Property of IBM
* "Restricted Materials of IBM"
* (C) Copyright IBM Corp. 2005. All Rights Reserved
*
===========================================================================
*/
package com.dovetail.jzos;

import java.io.*;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

public class Exec {

    private String command;
    private String[] cmdargs;
    private String[] envp;
    private File dir;

    private Process process;
    private class StdErrConsumer implements Runnable {
        private volatile boolean stopRequested = false;
        private int maxLines = 50;
        private List errorLines = new ArrayList();
        public void run() {
            try {
                BufferedReader reader = new BufferedReader(new InputStreamReader(process.getErrorStream()));
                String line;
                int lineCount = 0;

                while (!stopRequested && (line = reader.readLine()) != null) {
                    if (lineCount < maxLines) {
                        errorLines.add(line);
                        lineCount++;
                    }
                    healthTimerTask.update();
                }
            } catch (InterruptedIOException interIO) {
                System.out.println("StdErrorThread interrupted.");
            } catch (IOException ioe) {
                System.out.println("Unexpected exception reading ExternalProcess stderr output; remainder ignored: " + ioe.toString());
            }
        }
        public int getMaxLines() {
            return maxLines;
        }
        public void setMaxLines(int maxLines) {
            this.maxLines = maxLines;
        }
        public boolean isStopRequested() {
            return stopRequested;
        }
        public void setStopRequested(boolean stopRequested) {
            this.stopRequested = stopRequested;
        }
        public List getErrorLines() {
            return errorLines;
        }
    };

    private class  ProcessHealthTimerTask extends TimerTask {
        private volatile long lastResponseTime = 0;
        private int timeout = 10000;

        void setTimeout(int timeout) {
            this.timeout = timeout;
        }
        
        synchronized void update() {
            lastResponseTime = System.currentTimeMillis();
        }
        
        public void run() {
            System.out.println("TimerTask: running.");
            synchronized(this) {
                if (timeout > 0 && (lastResponseTime + timeout < System.currentTimeMillis())) {
                    System.out.println("TimeoutTask: Time out detected!");
                    Exec.this.destroy();
                }
            }
        }
    };
    
    boolean completed;
    boolean destroyed;
    BufferedReader outputReader;
    StdErrConsumer stdErrConsumer = new StdErrConsumer();
    Thread stdErrConsumerThread;
    ProcessHealthTimerTask healthTimerTask = new ProcessHealthTimerTask();
    Timer healthTimer = new Timer();

   /**
    * @see java.lang.Runtime#exec(java.lang.String)
    */
   public Exec(String command) {
       this(command, null, null);
   }

   /**
    * @see java.lang.Runtime#exec(java.lang.String, java.lang.String[])
    */
   public Exec(String command, String[] envp) {
       this(command, envp, null);
   }

   /**
    * @see java.lang.Runtime#exec(java.lang.String, java.lang.String[], java.io.File)
    */
   public Exec(String command, String[] envp, File dir) {
       this.command = command;
       this.envp = envp;
       this.dir = dir;
   }

   /**
    * @see java.lang.Runtime#exec(java.lang.String[])
    */
   public Exec(String[] cmdargs) {
       this(cmdargs, null, null);
   }

   /**
    * @see java.lang.Runtime#exec(java.lang.String[], java.lang.String[])
    */
   public Exec(String[] cmdargs, String[] envp) {
       this(cmdargs, envp, null);
   }
   
   /**
    * @see java.lang.Runtime#exec(java.lang.String[], java.lang.String[], java.io.File)
    */
   public Exec(String[] cmdargs, String[] envp, File dir) {
       this.cmdargs = cmdargs;
       this.envp = envp;
       this.dir = dir;
   }
   
   public static void main(String[] args) throws IOException {
       Exec exec;
       if (args.length == 0) {
           BufferedReader stdin = new BufferedReader(new InputStreamReader(System.in));
           System.out.println("Enter command: ");
           exec = new Exec(stdin.readLine());
       } else {
           exec = new Exec(args);
       }
       exec.run();

       String line;
       while ((line = exec.readLine()) != null) {
           System.out.println(line);
       }
       System.out.println("Return Code: " + exec.getReturnCode());
       for(Iterator iter = exec.getErrorLines().iterator(); iter.hasNext(); ) {
           System.err.println(iter.next());
       }
   }

   public List getErrorLines() {
       return stdErrConsumer.getErrorLines();
   }

   public void setMaxErrorLines(int maxErrorLines) {
       stdErrConsumer.setMaxLines(maxErrorLines);
   }

   public int getMaxErrorLines() {
       return stdErrConsumer.getMaxLines();
   }

   /**
    * Once all output is read (stderr and stdout), this method can be called to
    * retrieve the process exit value.
    */
   public int getReturnCode() {
       int rc = -1;
       if (!completed && !destroyed) {
           for(;;) {
               try {
                   rc = process.waitFor();
                   break;
               } catch (InterruptedException e) {
                   //loop until waitFor() succeeds
               }
           }
           healthTimer.cancel();
           try {
               stdErrConsumerThread.join();               
           } catch (InterruptedException e) {
           }
           completed = true;
       }
       return rc;
   }

   /**
    * Answers an OutputStream which is connected to the stdin input of the
    * external process.
    */
   public OutputStream getStdinStream() {

       return process.getOutputStream();
   }

   /**
    * O is disabled.
    */
   public void setTimeout(int timeout) {
       healthTimerTask.setTimeout(timeout);
   }

   public void consumeOutput(StringBuffer buf)
                      throws IOException {

       BufferedReader rdr = getOutputReader();
       String line;

       while ((line = rdr.readLine()) != null) {
           buf.append(line);
           buf.append('\n');
       }
   }

   public void consumeOutput()
                      throws IOException {

       BufferedReader rdr = getOutputReader();

       while (rdr.readLine() != null) {
       }
   }

     private void destroy() {
       synchronized(this) {
           process.destroy();
           destroyed = true;
       }
       healthTimer.cancel();
       stdErrConsumer.setStopRequested(true);
       stdErrConsumerThread.interrupt();
       try {
           stdErrConsumerThread.join();
       } catch (InterruptedException ie) {
           //eat
       }
   }

   public String readLine() throws IOException {

       String answer = getOutputReader().readLine();
       healthTimerTask.update();
       return answer;
   }

   public BufferedReader getOutputReader() {

       if (outputReader == null) {
           outputReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
       }

       return outputReader;
   } 
   public boolean isDestroyed() {
       return destroyed;
   }

   /**
    * Create (and start) the Runtime.exec process.  Also start the health
    * timer and stdErrConsumer Thread.
    */
   public void run() throws IOException {
       if (command != null) {
           process = Runtime.getRuntime().exec(command, envp, dir);
       } else {
           process = Runtime.getRuntime().exec(cmdargs, envp, dir);
       }
       healthTimer.schedule(healthTimerTask, 1000, 1000);
       stdErrConsumerThread = new Thread(stdErrConsumer);
       stdErrConsumerThread.start();
   }
}