/*
* %Z%%W% %I%
*
===========================================================================
* Licensed Materials - Property of IBM
* "Restricted Materials of IBM"
* (C) Copyright IBM Corp. 2005. All Rights Reserved
*
===========================================================================
*/
package com.dovetail.jzos;

/**
 * @author Stephen Goetze
 * 
 * Helper methods for converting to/from raw byte array data
 */
public class ByteUtil {
    
	/**
	 * Convert an int to four bytes (big-endian).
	 */
	public static byte[] intAsBytes(int i) {
	    byte[] bytes = new byte[4];
	    bytes[0] = (byte)((i >>> 24) & 0xFF);
	    bytes[1] = (byte)((i >>> 16) & 0xFF);
	    bytes[2] = (byte)((i >>> 8) & 0xFF);
	    bytes[3] = (byte)(i & 0xFF);
	
	    return bytes;
	}

	/**
	 * Convert up to 4 bytes into an int (big endian).
	 */
	public static int bytesAsInt(byte[] bytes) {
	    return bytesAsInt(bytes, 0, bytes.length);
	}
	
	public static int bytesAsInt(byte[] bytes, int offset, int length) {
	    if (length > 4) {
	        throw new RuntimeException("Only 4 or fewer bytes can fit into an int");
	    }
	    int val = 0;
	    for (int i=0; i<length; i++) {
	        val |= (bytes[offset+length-1-i] & 0xFF) << (i*8);
	    }	    
	    return val;
	}

	/**
	 * Unpack length bytes from the supplied byte array starting at offset.
	 * Each byte is assumed to contain two decimal digits (0-9).  The high order
	 * digit is a decimal order of magnitude greater than the low order digit.
	 * If isSigned is true, the final byte is assumed to contain a decimal digit
	 * in the high order nibble and a 'sign' hex digit in the low order nibble
	 * as follows:
	 * <ul>
	 * <li>'0xA','0xC','0xE','0xF' - Positive
	 * <li>'0xB','0xD' - Negative
	 * <li>Any other value - illegal
	 * </ul>
	 * @return long result
	 */
	public static long unpackLong(byte[] bytes, int offset, int length, boolean isSigned) {
	    long val = 0;
	    for(int i=0; i < length; i++) {
            int high_digit = (bytes[offset+i] >>> 4) & 0x0F;
            int low_digit = bytes[offset+i] & 0x0F;
	        if (i == length-1 && isSigned) {
	            val = val*10 + high_digit;
	            if (low_digit == 0x0D || low_digit == 0x0B) {
	                val = -val;
	            }
	        } else {
	            val = val*100 + high_digit*10 + low_digit;
	        }
	    }
	    return val;
	}
}
