//-----------------------------------------------------------------------------
// JsrLinePlot.java
//
// Many thanks to Kelvin R. Lawrence of IBM-Austin for his Paths sample
// from which I lifted all the "options" code.
//
// My contribution was to create make the program callable from another
// program, and add the logic to plot a variable number of lines
// for a variable number of points.
//
// Steve Ryder (JSR Systems) July, 2000. 
//-----------------------------------------------------------------------------
// A simple example of using the new path construction facilities of Java 2D.
// This version requires Java JDK level 1.2 (Beta4) or higher.
// Demonstrates use of these classes/interfaces:
//   Graphics2D
//   GeneralPath
// Original Author : Kelvin R Lawrence.     3rd-April-1998
//-----------------------------------------------------------------------------
import java.awt.* ;
import java.awt.event.* ;
import java.awt.geom.* ;

//---------------------------------------------------------------
// Class: JsrLinePlot
//
// Simple class that uses paths to draw line plots.  
//
//---------------------------------------------------------------
public class JsrLinePlot extends Frame implements ItemListener, ActionListener
{
  static final String COPYRIGHT = 
  "Copyright 2000, JSR Systems.  See: www.Jsrsys.com/copyright.";

  private String[] plotLabel;  // labels for each plotted line.
  private int[][]  plotValue;  // values for each line.
  private String[] plotOrder;  // 1-10 = value @ numPoint, 11-15 = array index
  private int      maxValue;   // maximum Value, all lines.
  private int      numPoint;   // number of Points to plot.
  private int      numLine;    // number of Lines to plot.
  private int      numPercent; // lastPercent length adjustment.
  static final Color[]  lineColor = 
         {Color.red,     Color.green, Color.blue,
          Color.orange,  Color.pink,  Color.yellow,
          Color.magenta, Color.cyan,  Color.white};
          
  static float[] [] dash =
      {
         { 30,  5, 20, 15 },
         { 20, 10,  5, 15 },
         { 10, 20, 15,  5 },
         {  5, 30, 20, 15 }
      }  ; // dashing patterns...


  JsrUtil               u = new JsrUtil(); // for pad Commands.
  JsrSortString sortOrder = new JsrSortString();
  JsrSysout        sysout = new JsrSysout();
  
  static final int DEFAULT_WIDTH =   2 ;
  static final int MAX_WIDTH     = 100 ;
  static final int MIN_WIDTH     =   1 ;

  private MenuBar  menuBar      ;
  private Menu     menuOptions  ;
  private Menu     menuEndings  ;
  private Menu     menuJoinings ;
  private Menu     menuWidths   ;
  private Menu     menuRender   ;
  private Menu     menuBeziers  ;

  private CheckboxMenuItem mitAntiAlias ;
  private CheckboxMenuItem mitClosePath ;
  private CheckboxMenuItem mitDashing   ;
  private CheckboxMenuItem mitOutline   ;

  private CheckboxMenuItem mitJoinMitre ;
  private CheckboxMenuItem mitJoinBevel ;
  private CheckboxMenuItem mitJoinRound ;

  private CheckboxMenuItem mitEndNone   ;
  private CheckboxMenuItem mitEndRound  ;
  private CheckboxMenuItem mitEndSquare ;

  private CheckboxMenuItem curEnding  ;
  private CheckboxMenuItem curJoining ;
  private CheckboxMenuItem curShape   ;
  private CheckboxMenuItem curColor   ;



  private MenuItem mitIncrease  ;
  private MenuItem mitDecrease  ;
  private MenuItem mitMaximum   ;
  private MenuItem mitMinimum   ;
  private MenuItem mitDefault   ;

  private Menu menuBackCol     ;
  private CheckboxMenuItem mitColGray  ;
  private CheckboxMenuItem mitColBlack ;

  private CheckboxMenuItem mitRenderQual ;
  private CheckboxMenuItem mitRenderFast ;
  private CheckboxMenuItem mitRenderDef  ;

  private boolean fAntiAlias ;
  private boolean fClosePath ;
  private boolean fDashing   ;
  private boolean fOutline   ;
  private boolean fControl   ;

  private int endStyle   ;
  private int joinStyle  ;
  private int lineWidth  ;

  private RenderingHints hints ;

  private Font labelFont; // move Font allocation out of paint
  // following added for "Filter";
  private boolean  selectSet;
  private Menu     menuSelect   ;
  private MenuItem mitSelect;
  private String   mitSelX = "*";
  private String[] selectItems   = new String[] {};
  //      if this string is null, then Select Menu won't be built.
  private int lastDwidth;
  private int lastDheight;
  private int lastDx;
  private int lastDy;

  //----------------------------------------------------------
  // Constructor
  //
  //----------------------------------------------------------
  public JsrLinePlot( String s )
  {
    super(s);  // to be sure frame has a title.

    enableEvents( AWTEvent.WINDOW_EVENT_MASK | AWTEvent.MOUSE_EVENT_MASK ) ;
    addWindowListener(new WindowExit());

    labelFont = new Font("Comic Sans Ms",Font.BOLD,12);
                        //SansSerif
    lastDwidth  = 0; // set default based on screen size
    lastDheight = 0;
    lastDx      = 0;
    lastDy      = 0;
    menuBar = new MenuBar() ;
    selectSet = false;  // will be turned on in first drawLinePlot call

    //----------------------------------------
    // Construct the "Options" menu, including
    // the Width sub menu and rendering sub
    // menu.
    //----------------------------------------
    menuOptions = new Menu( "Options" ) ;

    mitAntiAlias = new CheckboxMenuItem( "Use Anti aliasing" ) ;
    mitAntiAlias.setState(true); // turn on
    mitClosePath = new CheckboxMenuItem( "Close the path" ) ;
    mitDashing   = new CheckboxMenuItem( "Dashed lines" ) ;
    mitOutline   = new CheckboxMenuItem( "Draw outline" ) ;

            
    menuWidths  = new Menu( "Line Widths" ) ;

    mitIncrease = new MenuItem( "Increase (+5)" ) ;
    mitDecrease = new MenuItem( "Decrease (-5)" ) ;
    mitMaximum  = new MenuItem( "Maximum" ) ;
    mitMinimum  = new MenuItem( "Minimum" ) ;
    mitDefault  = new MenuItem( "Default" ) ;

    menuBackCol  = new Menu( "Background colour" ) ;
    mitColGray   = new CheckboxMenuItem( "Gray"  ) ;
    mitColBlack  = new CheckboxMenuItem( "Black" ) ;

    menuBackCol.add( mitColBlack ) ;
    menuBackCol.add( mitColGray  ) ;

    mitColBlack.addItemListener( this ) ;
    mitColGray.addItemListener( this ) ;

    menuRender  = new Menu( "Rendering" ) ;

    mitRenderDef   = new CheckboxMenuItem( "Default" ) ;
    mitRenderQual  = new CheckboxMenuItem( "Quality" ) ;
    mitRenderFast  = new CheckboxMenuItem( "Fastest" ) ;

    menuWidths.add( mitIncrease ) ;
    menuWidths.add( mitDecrease ) ;
    menuWidths.add( mitMaximum  ) ;
    menuWidths.add( mitMinimum  ) ;
    menuWidths.add( mitDefault  ) ;

    menuRender.add( mitRenderDef  ) ;
    menuRender.add( mitRenderQual ) ;
    menuRender.add( mitRenderFast ) ;

    menuOptions.add( mitAntiAlias ) ;
    menuOptions.add( mitClosePath ) ;
    menuOptions.add( mitDashing   ) ;
    menuOptions.add( mitOutline   ) ;
    menuOptions.add( menuWidths   ) ;
    menuOptions.add( menuRender   ) ;

    menuOptions.add( menuBackCol   ) ;

    mitAntiAlias.addItemListener( this ) ;
    mitClosePath.addItemListener( this ) ;
    mitDashing.addItemListener( this ) ;
    mitOutline.addItemListener( this ) ;

    mitIncrease.addActionListener( this ) ;
    mitDecrease.addActionListener( this ) ;
    mitMaximum.addActionListener( this ) ;
    mitMinimum.addActionListener( this ) ;
    mitDefault.addActionListener( this ) ;

    mitRenderDef.addItemListener( this ) ;
    mitRenderQual.addItemListener( this ) ;
    mitRenderFast.addItemListener( this ) ;

    //------------------------------
    // Construct the "Endings" menu.
    //------------------------------
    menuEndings = new Menu( "Endings" ) ;

    mitEndNone   = new CheckboxMenuItem( "None"   ) ;
    mitEndSquare = new CheckboxMenuItem( "Square" ) ;
    mitEndRound  = new CheckboxMenuItem( "Round"  ) ;

    menuEndings.add( mitEndNone   ) ;
    menuEndings.add( mitEndSquare ) ;
    menuEndings.add( mitEndRound  ) ;

    mitEndNone.addItemListener( this ) ;
    mitEndSquare.addItemListener( this ) ;
    mitEndRound.addItemListener( this ) ;

    //------------------------------
    //Construct the "Joinings" menu.
    //------------------------------
    menuJoinings = new Menu( "Joinings" ) ;

    mitJoinBevel = new CheckboxMenuItem( "Bevel" ) ;
    mitJoinMitre = new CheckboxMenuItem( "Mitre" ) ;
    mitJoinRound = new CheckboxMenuItem( "Round" ) ;

    menuJoinings.add( mitJoinBevel ) ;
    menuJoinings.add( mitJoinMitre ) ;
    menuJoinings.add( mitJoinRound ) ;

    mitJoinBevel.addItemListener( this ) ;
    mitJoinMitre.addItemListener( this ) ;
    mitJoinRound.addItemListener( this ) ;
 
    //---------------------------------------------------
    // Add the menus to the menu bar and add the menu bar
    // to the frame.
    //---------------------------------------------------
    menuBar.add( menuOptions  ) ;
    menuBar.add( menuEndings  ) ;
    menuBar.add( menuJoinings ) ;


    setMenuBar( menuBar ) ;

    //---------------------------
    // Initialise a few settings.
    //---------------------------

    fAntiAlias = true  ;
    fClosePath = false ;
    fDashing   = false ;
    fOutline   = false ;
    fControl   = true  ;

    endStyle  = BasicStroke.CAP_ROUND  ;
    joinStyle = BasicStroke.JOIN_MITER ;

    mitJoinMitre.setState( true ) ;
    mitEndRound.setState( true ) ;
    mitColBlack.setState( true ) ;

    curEnding  = mitEndRound  ;
    curJoining = mitJoinMitre ;
    curColor   = mitColBlack  ;

    lineWidth = DEFAULT_WIDTH ;

         //------------------------------------------
         // Setup our default set of rendering hints.
         //------------------------------------------
    hints = new RenderingHints( RenderingHints.KEY_ANTIALIASING
                              , RenderingHints.VALUE_ANTIALIAS_OFF ) ;

         hints.put( hints.KEY_RENDERING, hints.VALUE_RENDER_DEFAULT ) ;

    mitRenderDef.setState( true ) ;
  }

  //-------------------------------------------------------------------------
  // paint()
  //
  //-------------------------------------------------------------------------
  public void paint( Graphics g )
  {
    //System.out.println("Enter paint.");
    //System.out.println("Len="+numLine+"/"+numPoint);
    //System.out.println("Label[0]="+plotLabel[0]);
    //for (int i=0; i < plotValue[0].length; i++)
    //    System.out.print(plotValue[0][i]+"/");
    //System.out.println(".");
   
    Rectangle d = getBounds() ;
    Insets ins  = getInsets() ;
    System.out.println("Size(x/y/w/h)/Insets(l/t/r/b)="+
             d.x    +"/"+d.y+"/"+
             d.width+"/"+d.height+"/"+
           ins.left+"/"+ins.top+"/"+ins.right+"/"+ins.bottom);
    lastDwidth  = d.width;
    lastDheight = d.height;
    lastDx      = d.x;
    lastDy      = d.y;
    Graphics2D g2d = (Graphics2D) g ;
    g2d.setFont(labelFont);

    if ( curColor.equals( mitColBlack ))
    {
      g2d.setColor( Color.black ) ;
    }
    else
    {
      g2d.setColor( Color.lightGray ) ;
    }

    g2d.fillRect( 0,0,d.width,d.height ) ;

    //-------------------------------------
    // Set the appropriate rendering hints.
    //-------------------------------------

    if ( fAntiAlias )
    {
      //g2d.setRenderingHints( g2d.ANTIALIASING, g2d.ANTIALIAS_ON ) ;
      //hints.put( hints.KEY_ANTIALIASING, hints.VALUE_ANTIALIAS_ON ) ;
      g2d.setRenderingHint( RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON ) ;
    }
    else
    {
      //g2d.setRenderingHints( g2d.ANTIALIASING, g2d.ANTIALIAS_OFF ) ;
      //hints.put( hints.KEY_ANTIALIASING, hints.VALUE_ANTIALIAS_OFF ) ;
      g2d.setRenderingHint( RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF ) ;
    }

    //g2d.setRenderingHints( hints ) ;
    //System.out.println( hints.toString() ) ;

    //----------------------
    // Create a simple path.
    //----------------------
    long time1 = System.currentTimeMillis() ;

    int xStart = 100 ; int yStart = 100 ;

    // need to "convert" plotValue(s) to points scaled by d.width, d.height. JSR
    float beginY = d.height*15/16; // Y is up/down(Values).
    float beginX = d.width*1/16;  // X is across (numPoint)
    float scaleX = d.width*2/3/numPoint; // pixels across each iteration.
    float scaleY = (beginY-60)/maxValue;    // pixels each value?
    //    need to leave    60 pixels at top for title & menu lines.
    
    //sysout.display(scaleY+"/"+d.height+"/"+beginY+
      //         "<--scaleY/d.height/beginY");
    float thisX = 0;
    float thisY = 0; // to save last plotted point for label.
    //int lastY = 99999; // last label loc, to avoid overlap. (if going bottom to top)
    float lastY = 0; // if going top to bottom
    for (int iOrder = numLine-1; iOrder >= 0; iOrder--)
    {
      // do in order, lowest to highest so can adjust overlapping labels
      int i = u.getInt(plotOrder[iOrder].substring(10));
      //sysout.display("plotLabel[i]="+plotLabel[i].substring(0,1));
      if (!mitSelX.equals("*") &&
         (!mitSelX.equals(plotLabel[i].substring(0,mitSelX.length()))) )
         continue;  // break out of this iteration if not desired line
      
      //System.out.println("Plotting i="+i+" plotOrder: "+plotOrder[iOrder]);
      // loop to plot lines 
      GeneralPath path = new GeneralPath() ;
     
      // will plot the line(s) here...
      path.moveTo( beginX,beginY );
      for (int b=0; b < numPoint; b++)
      {
       thisX = beginX+(scaleX*(b+1));
       thisY = beginY-(scaleY*plotValue[i][b]);
       if (b == (numPoint-1)) thisX = thisX - (scaleX*(100-numPercent)/100);
       path.lineTo( thisX, thisY);
      }

      //-----------------------------------------
      // Close the path if requested by the user.
      //-----------------------------------------

      if ( fClosePath ) path.closePath() ;

      //-----------------------------------------------------------
      // Now make a few modifications to the  path and "stroke" it.
      //
      //-----------------------------------------------------------

      BasicStroke bs ;
  
      if ( fDashing )
      {
        int jDash = iOrder;
        while(jDash >= dash.length) jDash -= dash.length;
        bs = new BasicStroke( (float)lineWidth, endStyle, joinStyle,
                              10, dash[jDash], 0 ) ;
      }           //  mitreLimit, dash array, phase
      else
      {
        bs = new BasicStroke( (float)lineWidth, endStyle, joinStyle );
      }
  
      //-------------------------------
      // Now, finally, render the path.
      //-------------------------------

      int    jColor = iOrder;
      while (jColor >= lineColor.length) jColor -= lineColor.length;
      g2d.setColor(lineColor[jColor]);
      //System.out.println("i/jColor:"+i+"/"+jColor+lineColor[jColor].toString());
 
      if ( fOutline )
      {
        g2d.draw( bs.createStrokedShape( path ) ) ;
      }
      else
      {
        g2d.setStroke( bs ) ;
        g2d.draw( path ) ;
      }
      //if (lastY==0)sysout.display("lastY="+lastY+" thisY="+thisY);
      //if (thisY >= (lastY-15)) lastY = lastY - 15; // do this if going 
      //else                     lastY = thisY;      // bottom to top (-).
      if (thisY <= (lastY+15)) lastY = lastY + 15; // do this if going
      else                     lastY = thisY;      // top to bottom (+)
      g2d.drawString(u.padLeft(plotValue[i][numPoint-1],5)+" "+
                               plotLabel[i], thisX, lastY );
  

      // loop for each line ends here
    }
    long time2 = System.currentTimeMillis() ;
    //System.out.println((time2-time1)+"<-draw time (millis)       ");


  }  // end of paint routine


  //-------------------------------------------------------------------------
  // actionPerformed()
  //
  // Handle menu actions for which we have registered interest.
  //
  //-------------------------------------------------------------------------
  public void actionPerformed( ActionEvent aevt )
  {

    if ( aevt.getSource() instanceof MenuItem )
    {
      MenuItem item = (MenuItem)aevt.getSource() ;
      if ( item.equals( mitIncrease ) )
      {
        lineWidth += 5 ;

        if ( lineWidth >= MAX_WIDTH )
        {
          lineWidth = MAX_WIDTH ;
        }
      }
      else if ( item.equals( mitDecrease ) )
      {
        lineWidth -= 5 ;

        if ( lineWidth <= MIN_WIDTH )
        {
          lineWidth = MIN_WIDTH ;
        }
      }
      else if ( item.equals( mitMaximum ) )
      {
        lineWidth = MAX_WIDTH ;
      }
      else if ( item.equals( mitMinimum ) )
      {
        lineWidth = MIN_WIDTH ;
      }
      else if ( item.equals( mitDefault ) )
      {
        lineWidth = DEFAULT_WIDTH ;
      }
      else // all that is left is Select items...
      {
        String menuItem=aevt.getActionCommand();
        // sysout.display ("Menu: "+menuItem);
        mitSelX = menuItem;
      }

    }

    repaint() ;
  }

  //-------------------------------------------------------------------------
  // itemStateChanged()
  //
  // Handle menu actions for which we have registered interest.
  //
  //-------------------------------------------------------------------------
  public void itemStateChanged( ItemEvent ievt )
  {
    if ( ievt.getSource() instanceof CheckboxMenuItem )
    {
      CheckboxMenuItem item = (CheckboxMenuItem)ievt.getSource() ;

      if ( item.equals( mitAntiAlias ) )
      {
        fAntiAlias = item.getState() ;
      }
      else if ( item.equals( mitClosePath ) )
      {
        fClosePath = item.getState() ;
      }
      else if ( item.equals( mitDashing ) )
      {
        fDashing = item.getState() ;
      }
      else if ( item.equals( mitOutline ) )
      {
        fOutline = item.getState() ;
      }
      else if ( item.equals( mitEndRound ) )
      {
        curEnding.setState( false ) ;
        endStyle = BasicStroke.CAP_ROUND ;
        curEnding = item ;
        item.setState( true ) ;
      }
      else if ( item.equals( mitEndSquare ) )
      {
        curEnding.setState( false ) ;
        endStyle = BasicStroke.CAP_SQUARE ;
        curEnding = item ;
        item.setState( true ) ;
      }
      else if ( item.equals( mitEndNone ) )
      {
        curEnding.setState( false ) ;
        endStyle = BasicStroke.CAP_BUTT ;
        curEnding = item ;
        item.setState( true ) ;
      }
      else if ( item.equals( mitJoinBevel ) )
      {
        curJoining.setState( false ) ;
        joinStyle = BasicStroke.JOIN_BEVEL ;
        curJoining = item ;
        item.setState( true ) ;
      }
      else if ( item.equals( mitJoinMitre ) )
      {
        curJoining.setState( false ) ;
        joinStyle = BasicStroke.JOIN_MITER ;
        curJoining = item ;
        item.setState( true ) ;
      }
      else if ( item.equals( mitJoinRound ) )
      {
        curJoining.setState( false ) ;
        joinStyle = BasicStroke.JOIN_ROUND ;
        curJoining = item ;
        item.setState( true ) ;
      }
      else if ( item.equals( mitRenderDef ) )
      {
        //renderMode = Graphics2D.RENDER_DEFAULT ;
                  hints.put( hints.KEY_RENDERING, hints.VALUE_RENDER_DEFAULT ) ;
      }
      else if ( item.equals( mitRenderQual ) )
      {
         //renderMode = Graphics2D.RENDER_QUALITY ;
                        hints.put( hints.KEY_RENDERING, hints.VALUE_RENDER_QUALITY ) ;
      }
      else if ( item.equals( mitRenderFast ) )
      {
        //renderMode = Graphics2D.RENDER_SPEED ;
                  hints.put( hints.KEY_RENDERING, hints.VALUE_RENDER_SPEED ) ;
      }
      else if ( item.equals( mitColBlack ) || item.equals( mitColGray ))
      {
        curColor.setState( false ) ;
        curColor = item ;
        item.setState( true ) ;
      }

      repaint() ;
    }
  }

  // When winddow is closed.
    class WindowExit extends WindowAdapter
    {
        public void windowClosing(WindowEvent e)
        {
            Window w = e.getWindow();
            w.setVisible(false);
            w.dispose();
            sysout.display("JsrLinePlot--Exiting");
        }
    }
    // end class WindowExit
    // Note: this must be defined physically in the class
    //       BEFORE the first reference.  When I defined it at
    //       the end, it was not found, I moved it up here, all OK!
     
 
  // setFilter... sets up Filter Menu Item

  public void setFilter(String filterName, String[] lineSelect)
  {
     if (lineSelect.length > 0 &&
         this.selectSet == false) // add this menu item only if have array.
     {
       this.selectSet = true;  // will be turned on in first drawLinePlot call 
       this.selectItems = lineSelect;
       this.menuSelect = new Menu( filterName );
        
       for (int i=0; i < this.selectItems.length; i++)
       {
                this.mitSelect  = new MenuItem(this.selectItems[i]);
                this.mitSelect.addActionListener(this);
                this.menuSelect.add(this.mitSelect);
       }
       this.menuBar.add( this.menuSelect   );

       this.setMenuBar( this.menuBar ) ;
     }
    
  }

  // drawLinePlot() entry from another class

  public void drawLinePlot (String titleTxt, String[] lineLabel, int[][] lineValue,
                            int lastPercent)
  {
     sysout.display("Enter JsrLinePlot: "+
                    "Len="+lineLabel.length+"/"+lineValue.length+
                       "/"+lineValue[0].length+"/"+
                       "%="+lastPercent+"/"+
                       lineLabel[0]);
     this.setTitle(titleTxt);

     // must set this. values to get correct instance of JsrLinePlot for paint.
     // without the this. is setting instance created by calling class.  
     this.numPoint  = lineValue[0].length;
     this.numPercent = lastPercent;
     this.numLine   = lineValue.length;
     this.plotLabel = lineLabel; // sets whole array
     this.plotValue = lineValue; // sets whole array
     this.maxValue  = 0;
     this.plotOrder = new String[this.numLine];
     //System.out.println("Len="+this.numLine+"/"+this.numPoint+"/"+this.plotLabel[0]);
     //System.out.println("lenOrder="+this.plotOrder.length);
     for (int i=0; i < this.numLine; i++)
     {
       for (int b=0; b < this.numPoint; b++)
       {
        //System.out.print(this.plotValue[i][b]+"/");
        if (this.maxValue < this.plotValue[i][b])
            this.maxValue = this.plotValue[i][b];
       }
       //System.out.print("numPoint-1="+(this.numPoint-1)+"/");
       //System.out.println("lastValue="+this.plotValue[i][this.numPoint-1]);
       //System.out.println(u.padZero(this.plotValue[i][this.numPoint-1],10)+u.padZero(i,5));
       
       this.plotOrder[i] = u.padZero(this.plotValue[i][this.numPoint-1],10)+u.padZero(i,5);
       //System.out.println("plotOrder[...]: "+this.plotOrder[i]);
     }
     //System.out.println("\r\nMax="+this.maxValue+".");
     try{sortOrder.sort(this.plotOrder);} // sorts array in place.
        catch (Exception Sort) {System.out.println("Sort Error in JsrLinePlot!");}
     //System.out.println("Sort: "+this.plotOrder[0]+"/"+this.plotOrder[1]); 
      
     Toolkit tlkt = Toolkit.getDefaultToolkit() ;
     Dimension dim = tlkt.getScreenSize() ;
     if (lastDwidth  == 0) lastDwidth  = dim.width * 7/8;
     if (lastDheight == 0) lastDheight = dim.height* 7/8;
     if (lastDx      == 0) lastDx      = dim.width * 1/16;
     if (lastDy      == 0) lastDy      = dim.height* 1/16;
     this.setBounds( lastDx,lastDy,lastDwidth,lastDheight );
     // above will leave a 1/16 border around a window that fills 7/8 of the screen.
     // or set to last size & position used.
     this.show() ;
     //return; -- closes with window listener.
  }
//--end of class JsrLinePlot  
}